<?php
/**
 * Migration: Create Transactions Tables
 * Creates tables for quotations, deliveries, and invoices modules
 */

require_once __DIR__ . '/../config/db.php';

try {
    // Create quotations table
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS `quotations` (
            `id` int(11) NOT NULL AUTO_INCREMENT,
            `quote_number` varchar(50) NOT NULL UNIQUE,
            `customer_id` int(11) DEFAULT NULL,
            `total_amount` decimal(10,2) NOT NULL DEFAULT 0.00,
            `status` enum('active','expired','converted','cancelled') DEFAULT 'active',
            `notes` text,
            `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
            `updated_at` timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            `valid_until` date NOT NULL,
            `created_by` int(11) NOT NULL,
            PRIMARY KEY (`id`),
            KEY `customer_id` (`customer_id`),
            KEY `created_by` (`created_by`),
            KEY `status` (`status`),
            CONSTRAINT `quotations_customer_fk` FOREIGN KEY (`customer_id`) REFERENCES `customers` (`id`) ON DELETE SET NULL,
            CONSTRAINT `quotations_creator_fk` FOREIGN KEY (`created_by`) REFERENCES `users` (`id`) ON DELETE CASCADE
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    ");
    
    // Create quotation_items table
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS `quotation_items` (
            `id` int(11) NOT NULL AUTO_INCREMENT,
            `quotation_id` int(11) NOT NULL,
            `product_id` int(11) NOT NULL,
            `quantity` decimal(10,2) NOT NULL,
            `unit_price` decimal(10,2) NOT NULL,
            `total_price` decimal(10,2) NOT NULL,
            `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (`id`),
            KEY `quotation_id` (`quotation_id`),
            KEY `product_id` (`product_id`),
            CONSTRAINT `quotation_items_quotation_fk` FOREIGN KEY (`quotation_id`) REFERENCES `quotations` (`id`) ON DELETE CASCADE,
            CONSTRAINT `quotation_items_product_fk` FOREIGN KEY (`product_id`) REFERENCES `products` (`id`) ON DELETE CASCADE
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    ");
    
    // Create deliveries table
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS `deliveries` (
            `id` int(11) NOT NULL AUTO_INCREMENT,
            `delivery_number` varchar(50) NOT NULL UNIQUE,
            `customer_id` int(11) DEFAULT NULL,
            `sale_id` int(11) DEFAULT NULL,
            `delivery_address` text NOT NULL,
            `delivery_phone` varchar(20) NOT NULL,
            `scheduled_date` datetime NOT NULL,
            `delivered_date` datetime DEFAULT NULL,
            `status` enum('pending','in-transit','delivered','cancelled') DEFAULT 'pending',
            `assigned_to` int(11) DEFAULT NULL,
            `delivery_fee` decimal(10,2) DEFAULT 0.00,
            `payment_method` enum('cash','paid','credit') DEFAULT 'cash',
            `delivery_notes` text,
            `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
            `updated_at` timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            `created_by` int(11) NOT NULL,
            PRIMARY KEY (`id`),
            KEY `customer_id` (`customer_id`),
            KEY `sale_id` (`sale_id`),
            KEY `assigned_to` (`assigned_to`),
            KEY `status` (`status`),
            KEY `scheduled_date` (`scheduled_date`),
            CONSTRAINT `deliveries_customer_fk` FOREIGN KEY (`customer_id`) REFERENCES `customers` (`id`) ON DELETE SET NULL,
            CONSTRAINT `deliveries_sale_fk` FOREIGN KEY (`sale_id`) REFERENCES `sales` (`id`) ON DELETE SET NULL,
            CONSTRAINT `deliveries_assignee_fk` FOREIGN KEY (`assigned_to`) REFERENCES `users` (`id`) ON DELETE SET NULL,
            CONSTRAINT `deliveries_creator_fk` FOREIGN KEY (`created_by`) REFERENCES `users` (`id`) ON DELETE CASCADE
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    ");
    
    // Create invoices table
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS `invoices` (
            `id` int(11) NOT NULL AUTO_INCREMENT,
            `invoice_number` varchar(50) NOT NULL UNIQUE,
            `customer_id` int(11) DEFAULT NULL,
            `sale_id` int(11) DEFAULT NULL,
            `subtotal` decimal(10,2) NOT NULL DEFAULT 0.00,
            `tax_amount` decimal(10,2) NOT NULL DEFAULT 0.00,
            `total_amount` decimal(10,2) NOT NULL DEFAULT 0.00,
            `status` enum('draft','pending','paid','overdue','cancelled') DEFAULT 'pending',
            `issue_date` date NOT NULL,
            `due_date` date NOT NULL,
            `paid_date` datetime DEFAULT NULL,
            `payment_terms` varchar(50) DEFAULT 'net_30',
            `notes` text,
            `customer_email` varchar(255) DEFAULT NULL,
            `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
            `updated_at` timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            `created_by` int(11) NOT NULL,
            PRIMARY KEY (`id`),
            KEY `customer_id` (`customer_id`),
            KEY `sale_id` (`sale_id`),
            KEY `created_by` (`created_by`),
            KEY `status` (`status`),
            KEY `due_date` (`due_date`),
            CONSTRAINT `invoices_customer_fk` FOREIGN KEY (`customer_id`) REFERENCES `customers` (`id`) ON DELETE SET NULL,
            CONSTRAINT `invoices_sale_fk` FOREIGN KEY (`sale_id`) REFERENCES `sales` (`id`) ON DELETE SET NULL,
            CONSTRAINT `invoices_creator_fk` FOREIGN KEY (`created_by`) REFERENCES `users` (`id`) ON DELETE CASCADE
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    ");
    
    // Create invoice_items table
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS `invoice_items` (
            `id` int(11) NOT NULL AUTO_INCREMENT,
            `invoice_id` int(11) NOT NULL,
            `product_id` int(11) NOT NULL,
            `quantity` decimal(10,2) NOT NULL,
            `unit_price` decimal(10,2) NOT NULL,
            `total_price` decimal(10,2) NOT NULL,
            `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (`id`),
            KEY `invoice_id` (`invoice_id`),
            KEY `product_id` (`product_id`),
            CONSTRAINT `invoice_items_invoice_fk` FOREIGN KEY (`invoice_id`) REFERENCES `invoices` (`id`) ON DELETE CASCADE,
            CONSTRAINT `invoice_items_product_fk` FOREIGN KEY (`product_id`) REFERENCES `products` (`id`) ON DELETE CASCADE
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    ");
    
    // Create auto-increment triggers for generating numbers
    $pdo->exec("
        CREATE TRIGGER IF NOT EXISTS `quotation_number_generator` 
        BEFORE INSERT ON `quotations` 
        FOR EACH ROW 
        BEGIN 
            IF NEW.quote_number IS NULL OR NEW.quote_number = '' THEN
                SET NEW.quote_number = CONCAT('QUO-', YEAR(NOW()), '-', LPAD(COALESCE((SELECT MAX(CAST(SUBSTRING(quote_number, 9) AS UNSIGNED)) FROM quotations WHERE quote_number LIKE CONCAT('QUO-', YEAR(NOW()), '-%')), 0) + 1, 4, '0'));
            END IF;
        END
    ");
    
    $pdo->exec("
        CREATE TRIGGER IF NOT EXISTS `delivery_number_generator` 
        BEFORE INSERT ON `deliveries` 
        FOR EACH ROW 
        BEGIN 
            IF NEW.delivery_number IS NULL OR NEW.delivery_number = '' THEN
                SET NEW.delivery_number = CONCAT('DEL-', YEAR(NOW()), '-', LPAD(COALESCE((SELECT MAX(CAST(SUBSTRING(delivery_number, 9) AS UNSIGNED)) FROM deliveries WHERE delivery_number LIKE CONCAT('DEL-', YEAR(NOW()), '-%')), 0) + 1, 4, '0'));
            END IF;
        END
    ");
    
    $pdo->exec("
        CREATE TRIGGER IF NOT EXISTS `invoice_number_generator` 
        BEFORE INSERT ON `invoices` 
        FOR EACH ROW 
        BEGIN 
            IF NEW.invoice_number IS NULL OR NEW.invoice_number = '' THEN
                SET NEW.invoice_number = CONCAT('INV-', YEAR(NOW()), '-', LPAD(COALESCE((SELECT MAX(CAST(SUBSTRING(invoice_number, 9) AS UNSIGNED)) FROM invoices WHERE invoice_number LIKE CONCAT('INV-', YEAR(NOW()), '-%')), 0) + 1, 4, '0'));
            END IF;
        END
    ");
    
    echo "SUCCESS: All transaction tables created successfully!\n";
    echo "Tables created:\n";
    echo "- quotations\n";
    echo "- quotation_items\n";
    echo "- deliveries\n";
    echo "- invoices\n";
    echo "- invoice_items\n";
    echo "- Auto-increment triggers for document numbers\n";
    
} catch (PDOException $e) {
    echo "ERROR: " . $e->getMessage() . "\n";
    exit(1);
}
?> 